/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once


/////////////////////// Qt includes
#include <QString>
#include <QDomElement>


/////////////////////// Local includes
#include "exportimportconfig.h"
#include "IsotopicData.hpp"
#include "Ponderable.hpp"


namespace MsXpS
{


namespace libXpertMass
{

/*
 Bitwise stuff (from StackOverflow)

 It is sometimes worth using an enum to name the bits:

 enum ThingFlags = {
 ThingMask  = 0x0000,
 ThingFlag0 = 1 << 0,
 ThingFlag1 = 1 << 1,
 ThingError = 1 << 8,
 }

 Then use the names later on. I.e. write

 thingstate |= ThingFlag1;
 thingstate &= ~ThingFlag0;
 if (thing & ThingError) {...}

 to set, clear and test. This way you hide the magic numbers from the rest of
 your code.
 */
enum class FormulaSplitResult
{
  NOT_SET             = 0x0000,
  FAILURE             = 1 << 0, //!< The splitting work failed.
  HAS_PLUS_COMPONENT  = 1 << 1, //!< The action formula a plus component
  HAS_MINUS_COMPONENT = 1 << 2, //!< The action formula a minus component
  HAS_BOTH_COMPONENTS =
    (HAS_PLUS_COMPONENT |
     HAS_MINUS_COMPONENT) //!< The action formula had both component types
};

class DECLSPEC Formula
{
  public:
  Formula(const QString &formula = QString());
  Formula(const Formula &other);

  virtual ~Formula();

  void setFormula(const QString &formula);
  void setFormula(const Formula &formula);

  void appendFormula(const QString &formula);

  virtual QString toString() const;

  void setForceCountIndex(bool forceCountIndex);
  void clear();

  const std::map<QString, double> getSymbolCountMap() const;

  std::size_t accountFormula(const QString &text,
                             IsotopicDataCstSPtr isotopic_data_csp,
                             double times = 1);

  virtual Formula &operator=(const Formula &other);
  virtual bool operator==(const Formula &other) const;
  virtual bool operator!=(const Formula &other) const;

  static QChar actions(const QString &formula);
  QChar actions() const;

  static bool checkSyntax(const QString &formula, bool forceCountIndex = false);
  bool checkSyntax() const;
  bool checkSyntaxRegExp(const QString &formula, bool forceCountIndex = false);

  virtual bool validate(IsotopicDataCstSPtr isotopic_data_csp);

  virtual bool validate(IsotopicDataCstSPtr isotopic_data_csp, bool, bool);

  virtual bool accountMasses(IsotopicDataCstSPtr isotopic_data_csp,
                             double *mono = Q_NULLPTR,
                             double *avg  = Q_NULLPTR,
                             double times = 1);

  virtual bool accountMasses(IsotopicDataCstSPtr isotopic_data_csp,
                             Ponderable *ponderable,
                             double times = 1);

  bool accountSymbolCounts(IsotopicDataCstSPtr isotopic_data_csp, int);
  QString elementalComposition(std::vector<std::pair<QString, double>>
                                 *symbol_count_pairs_p = nullptr) const;

  double totalAtoms() const;
  double totalIsotopes(IsotopicDataCstSPtr isotopic_data_csp) const;
  double symbolCount(const QString &symbol) const;

  bool hasNetMinusPart();

  bool renderXmlFormulaElement(const QDomElement &);

  protected:
  QString m_formula;

  QString m_plusFormula;

  QString m_minusFormula;

  // Map that relates the chemical element symbols encounted in formula(e) to
  // their count in the formula(e). The count can be a double or an int.
  std::map<QString, double> m_symbolCountMap;

  IsotopicDataCstSPtr mcsp_isotopicData = nullptr;

  //! Tell if elements present a single time need to have an index nonetheless
  /*!
    When a formula is like H2O, if true, then the formula does not validate
    if it is not defined like so: H2O1 (note the O1).
    */
  bool m_forceCountIndex = false;

  void setPlusFormula(const QString &formula);
  const QString &plusFormula() const;

  void setMinusFormula(const QString &formula);
  const QString &minusFormula() const;

  int removeTitle();
  int removeSpaces();

  int splitActionParts(IsotopicDataCstSPtr isotopic_data_csp,
                       double times = 1,
                       bool store   = false,
                       bool reset   = false);

  double accountSymbolCountPair(const QString &symbol, double count = 1);
};

} // namespace libXpertMass

} // namespace MsXpS
